/* batch_Baseball-Homerun-1.jsx
   Saves PNG sequences under .../<league>/
   Animation folder & file base: LEAGUE_teamId_ABBR_<RUN_TAG>-<runNum><ANIM_TAIL>
   Example: MLB_12312_PHI_R-1_1_3
*/

(function () {
  // ---------- utils ----------
  function env(k,d){ var v=$.getenv(k); return (v===null||v===undefined||v==="")?d:v; }
  function trim(s){ var t=String(s||""); while(/^[ \t\r\n]/.test(t)) t=t.substring(1); while(/[ \t\r\n]$/.test(t)) t=t.substring(0,t.length-1); return t; }
  function toLower(s){ return String(s||"").toLowerCase(); }
  function clamp01(v){ return Math.max(0, Math.min(1, v)); }
  function sanitize(s){ var bad='\\/:*?"<>|', t=String(s||""), out="", i, ch; for(i=0;i<t.length;i++){ ch=t.charAt(i); out+=(bad.indexOf(ch)>=0?"-":ch);} return out; }
  function numOr(v,def){ var n=parseFloat(v); return (isFinite(n) ? n : def); }
  function safeColor(c){
    if (!c || c.length!==3) return [0,0,0];
    function cl(x){ return (x<0?0:(x>1?1:x)); }
    var r=numOr(c[0],0), g=numOr(c[1],0), b=numOr(c[2],0);
    return [cl(r),cl(g),cl(b)];
  }
  function rgb01(r,g,b){ return [clamp01(numOr(r,0)/255), clamp01(numOr(g,0)/255), clamp01(numOr(b,0)/255)]; }
  function trySet(p,v){ if (!p) return false; try{ p.setValue(v); return true; }catch(e){ return false; } }

  // Color helpers
  function isBlackish(c){ return (!c || c.length!==3 || (c[0]<=0.05 && c[1]<=0.05 && c[2]<=0.05)); }
  function lightGrey(){ return [0.7,0.7,0.7]; }
  function smartPrimary(p,s){ return isBlackish(p) ? (isBlackish(s)?lightGrey():s) : p; }
  function smartSecondary(p,s){ return isBlackish(s) ? (isBlackish(p)?lightGrey():p) : s; }

  // Unhide / setup
  function unhideAll(comp){
    try{ comp.hideShyLayers=false; }catch(e){}
    for (var i=1;i<=comp.numLayers;i++){
      var L=comp.layer(i);
      try{ L.enabled=true; }catch(e){}
      try{ L.shy=false; }catch(e){}
      try{ if (L.source && (L.source instanceof CompItem)) unhideAll(L.source); }catch(e){}
    }
  }
  function setupComp(comp){
    comp.width = Math.round(comp.width);
    comp.height = Math.round(comp.height);
    comp.motionBlur = false;
  }

  // ---------- env ----------
  var PROJECT   = env("AE_PROJECT", null);
  var CSV_PATH  = env("AE_CSV", null);
  var COMP_NAME = env("AE_COMP","HomeRun");

  var TEAMNAME_LAYER      = env("AE_TEAMNAME_LAYER","TeamName");
  var TEAMLOGO_LAYER      = env("AE_TEAMLOGO_LAYER","TeamLogo");
  var STREAKS_LAYER       = env("AE_STREAKS_LAYER","Streaks");
  var LARGEBASEBALL_LAYER = env("AE_LARGEBASEBALL_LAYER","LargeBaseball");
  var BASEBALL_PATH       = env("AE_BASEBALL_PATH","");

  var LOGO_DIR  = env("AE_LOGO_DIR","");
  var LOGO_EXTS = env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd").split(",");

  var LEAGUE    = env("AE_LEAGUE","MLB");
  var TEAMS_LIST= env("AE_TEAMS","");
  var LIMIT_STR = env("AE_LIMIT","");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = env("AE_OUTDIR","");
  var PATH_TPL  = env("AE_PATH_TEMPLATE","{league}"); // league-only
  var OM_TPL    = env("AE_OM_TEMPLATE","PNG Sequence");

  // naming controls
  var RUN_TAG   = env("AE_RUN_TAG","R");
  var ANIM_TAIL = env("AE_ANIM_TAIL","_1_3");
  function normTail(s){ s=String(s||""); if(!s) return ""; if(s.charAt(0)!=="_") s="_"+s; return sanitize(s); }

  // ---------- CSV ----------
  function openRead(path){ var f=new File(path); if(!f.exists) fail("File not found: "+path); f.open("r"); var s=f.read(); f.close(); return s; }
  function fail(msg){ alert(msg); throw new Error(msg); }
  function parseCSV(txt){
    var raw=txt.split(/\r\n|\n|\r/), lines=[], i; for(i=0;i<raw.length;i++){ var L=raw[i]; if(L && !/^\s*$/.test(L)) lines.push(L); }
    var rows=[], c; for(i=0;i<lines.length;i++){ var line=lines[i]; var cells=(line.indexOf("\t")!==-1? line.split("\t") : line.split(",")); 
      for(c=0;c<cells.length;c++){ var cell=trim(cells[c]); if(cell.charAt(0)=='"' && cell.charAt(cell.length-1)=='"') cell=cell.substring(1,cell.length-1); cells[c]=cell; }
      rows.push(cells);
    }
    if(rows.length<2) fail("CSV has no data."); return rows;
  }
  function headerIdx(h){
    var m={}, i; for(i=0;i<h.length;i++) m[toLower(h[i])]=i;
    function need(x){ if(m[x]===undefined) fail("Missing column: "+x); }
    need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
    return m;
  }
  function buildTeams(rows){
    var h=rows[0], idx=headerIdx(h), out=[], i;
    var leagueIdx = idx["league"];
    var nameIdx   = (idx["name"]!==undefined ? idx["name"] : idx["displayname"]);
    var tidIdx    = (idx["espn_team_id"]!==undefined ? idx["espn_team_id"]
                   : (idx["team_id"]!==undefined ? idx["team_id"]
                   : (idx["id"]!==undefined ? idx["id"] : undefined)));
    for(i=1;i<rows.length;i++){
      var r=rows[i]; var ab=trim(r[idx["abbreviation"]]); if(!ab) continue;
      out.push({
        abbr: ab,
        league: (leagueIdx!==undefined ? trim(rows[i][leagueIdx]) : "MLB"),
        name:   (nameIdx!==undefined   ? trim(rows[i][nameIdx])   : ab),
        team_id: (tidIdx!==undefined   ? trim(rows[i][tidIdx])    : ""),
        primary:   rgb01(r[idx["r"]],  r[idx["g"]],  r[idx["b"]]),
        secondary: rgb01(r[idx["r2"]], r[idx["g2"]], r[idx["b2"]])
      });
    }
    return out;
  }
  function pickTeams(all){
    var res=[], i;
    if (TEAMS_LIST){
      var want={}, parts=TEAMS_LIST.split(",");
      for(i=0;i<parts.length;i++){ want[trim(parts[i])] = true; }
      for(i=0;i<all.length;i++){ if(want[all[i].abbr]) res.push(all[i]); }
    } else if (LEAGUE){
      for(i=0;i<all.length;i++){ if(toLower(all[i].league)===toLower(LEAGUE)) res.push(all[i]); }
    } else { res = all.slice(0); }
    if (LIMIT && res.length>LIMIT) res = res.slice(0, LIMIT);
    return res;
  }

  // ---------- AE helpers ----------
  function findComp(name){ for(var i=1;i<=app.project.numItems;i++){ var it=app.project.item(i); if(it instanceof CompItem && it.name===name) return it; } return null; }
  function getLayer(comp, name){ try{ return comp.layer(name); }catch(e){ return null; } }

  function setTextLayerProperties(layer, text, fillColor, strokeColor, strokeWidth){
    var st=layer.property("Source Text"); if(!st) return false;
    var td=st.value;
    if (text !== null) td.text = String(text||"");
    td.applyFill = true;
    td.fillColor = safeColor(fillColor);
    if (strokeColor !== null && strokeWidth !== null){
      td.applyStroke = true;
      td.strokeColor = strokeColor;
      td.strokeWidth = strokeWidth;
    }
    try{ st.setValue(td); return true; }catch(e){ return false; }
  }

  function setTintEffectColors(layer, color){
    var fx = layer.property("Effects");
    if (!fx) return false;
    var updated = false;
    for (var i=1; i<=fx.numProperties; i++){
      var effect = fx.property(i);
      if (effect.matchName === "ADBE Tint"){
        // locale-agnostic: set every COLOR sub-prop & force Amount 100
        for (var j=1; j<=effect.numProperties; j++){
          var p = effect.property(j);
          if (p && p.propertyValueType === PropertyValueType.COLOR){
            try { p.setValue(safeColor(color)); updated = true; } catch(e){}
          }
        }
        var amount = effect.property("Amount to Tint") || effect.property("Amount");
        if (amount) try { amount.setValue(100); } catch(e){}
      }
    }
    return updated;
  }

  function replaceLogo(comp, layerName, abbr){
    var lyr = comp.layer(layerName); if (!lyr || !LOGO_DIR) return;
    var logoFile = null;
    for (var i=0;i<LOGO_EXTS.length;i++){
      var f = new File(LOGO_DIR + "/" + abbr + "." + trim(LOGO_EXTS[i]));
      if (f.exists){ logoFile = f; break; }
    }
    if (!logoFile) return;
    var io = new ImportOptions(logoFile); if (!io.canImportAs(ImportAsType.FOOTAGE)) return;
    var footage = app.project.importFile(io);
    lyr.replaceSource(footage, false);
  }

  function replaceBaseball(comp, layerName, runNum){
    var lyr = comp.layer(layerName); if (!lyr || !BASEBALL_PATH) return;
    var baseballFile = new File(BASEBALL_PATH + "-" + runNum + ".png");
    if (!baseballFile.exists) { alert("Baseball file not found: " + baseballFile.fsName); return; }
    var io = new ImportOptions(baseballFile); if (!io.canImportAs(ImportAsType.FOOTAGE)) return;
    var footage = app.project.importFile(io);
    lyr.replaceSource(footage, false);
  }

  // Robust PNG OM applier
  function applyPNGOMOrDie(om, preferred){
    var want = []; if (preferred) want.push(preferred);
    want.push("PNG Sequence","PNG Sequence with Alpha","PNG","PNG-sequence");
    var avail = []; try{ avail = om.templates || []; }catch(e){}
    function cieq(a,b){ return String(a).toLowerCase()===String(b).toLowerCase(); }
    for (var i=0;i<want.length;i++){
      for (var j=0;j<avail.length;j++){
        if (cieq(avail[j], want[i])){ try{ om.applyTemplate(avail[j]); return avail[j]; }catch(e){} }
      }
    }
    for (var k=0;k<avail.length;k++){
      if (String(avail[k]).toLowerCase().indexOf("png")!==-1){
        try{ om.applyTemplate(avail[k]); return avail[k]; }catch(e){}
      }
    }
    throw new Error("No PNG Output Module template found. Available: ["+(avail.join(", ")||"none")+"]");
  }

  // ---------- run ----------
  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("MLB Animation - PNG Sequences");

  function fail(msg){ alert(msg); throw new Error(msg); }

  if(!PROJECT) fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) fail("AE_CSV env not set.");
  var rows=parseCSV(openRead(CSV_PATH)), teams=buildTeams(rows), todo=pickTeams(teams);
  if(!todo.length) fail("No teams matched.");

  var rootComp=findComp(COMP_NAME); if(!rootComp) fail("Comp not found: "+COMP_NAME);
  unhideAll(rootComp);
  setupComp(rootComp);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  if(!rootOut.exists) rootOut.create();

  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    var leagueLabel = t.league || LEAGUE || "MLB";
    var teamIdSafe  = sanitize(t.team_id || "NA");
    var abbrSafe    = sanitize(t.abbr);

    var prim = smartPrimary(safeColor(t.primary),   safeColor(t.secondary));
    var sec  = smartSecondary(safeColor(t.primary), safeColor(t.secondary));
    var black= [0,0,0];

    // Output path .../<league>/
    var sub = PATH_TPL.replace("{league}", sanitize(leagueLabel))
                      .replace("{abbr}",   abbrSafe);
    var outFolder = new Folder(rootOut.fsName + "/" + sub);
    if (!outFolder.exists) outFolder.create();

    for (var runNum=1; runNum<=4; runNum++){
      // 1) Logo
      replaceLogo(rootComp, TEAMLOGO_LAYER, t.abbr);

      // 2) TeamName text (smart fill, thin dark stroke)
      var nameLy = getLayer(rootComp, TEAMNAME_LAYER);
      if (nameLy){ setTextLayerProperties(nameLy, (t.name||t.abbr).toUpperCase(), prim, black, 0.5); }

      // 3) Streaks tint → primary (force 100%)
      var streaksLy = getLayer(rootComp, STREAKS_LAYER);
      if (streaksLy){ setTintEffectColors(streaksLy, prim); }

      // 4) Baseball image variant
      replaceBaseball(rootComp, LARGEBASEBALL_LAYER, runNum);

      // Animation folder & base filename:
      // LEAGUE_teamId_ABBR_<RUN_TAG>-<runNum><ANIM_TAIL>
      var animName = sanitize(leagueLabel) + "_" + teamIdSafe + "_" + abbrSafe +
                     "_" + sanitize(RUN_TAG) + "-" + runNum + normTail(ANIM_TAIL);

      var animFolder = new Folder(outFolder.fsName + "/" + animName);
      if (!animFolder.exists) animFolder.create();

      // Output file base (AE adds numbering + .png)
      var outFile = File(animFolder.fsName + "/" + animName);

      // Render one run at a time; ensure PNG OM
      var rqItem = app.project.renderQueue.items.add(rootComp);
      var om=rqItem.outputModule(1);
      applyPNGOMOrDie(om, OM_TPL);
      om.file = outFile;

      app.project.renderQueue.render();

      // clear queue between runs
      while(app.project.renderQueue.numItems>0){ app.project.renderQueue.item(1).remove(); }
    }
  }

  app.endUndoGroup();
  if (env("AE_QUIT","1")==="1") app.quit();
})();
